#region References

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;
using gov.va.med.vbecs.Common;
using System.Text;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

#endregion

namespace gov.va.med.vbecs.GUI.controls
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/13/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///GUI User Control allows users to enter Special Testing results for a blood unit
	///</summary>

	#endregion

	public class SpecialTesting : GUI.controls.BaseControl
	{
		#region Variables

		/// <summary>
		/// Efficiently store antigen types (NEG or POS)
		/// </summary>
		private const string _antigenTypeNeg = "NEG";
		private const string _antigenTypePos = "POS";

		private string divisionCode = string.Empty;

		/// <summary>
		/// The type of the antigen that was changed:
		/// POS or NEG
		/// </summary>
		private string _antigenChangedType = string.Empty;

		private BOL.BloodUnitTest cmvNeg;
		private BOL.BloodUnitTest sickleCellNeg;

		private System.Collections.ArrayList _bloodUnitAntigens;
		private System.Collections.ArrayList _preLoadedBloodUnitAntigens;
		
		private Guid bloodUnitGUID = Guid.Empty;

		/// <summary>
		/// The type id of the antigen that was changed
		/// </summary>
		private int _antigenChangedTypeId;

		/// <summary>
		/// Has the changed antigen been checked or un-checked?
		/// </summary>
		private bool _antigenChangedChecked = false;

		private bool _isLoading = false;
		private bool antigensChanged = false;

		/// <summary>
		/// Has the object been changed
		/// </summary>
		private bool _isDirty = false;

		/// <summary>
		/// Do we allow updates, or adds only
		/// </summary>
		private bool _allowUpdate = true;

		/// <summary>
		/// Object is dirty changed event
		/// </summary>
		public event EventHandler IsDirtyChanged;

		/// <summary>
		/// Event to be raised when an antigen is updated
		/// </summary>
		public event EventHandler AntigenUpdated;

		#endregion

		#region Controls

		private System.Windows.Forms.GroupBox grpSpecial;
		private System.Windows.Forms.CheckBox cbSickleCellScreen;
		private System.Windows.Forms.CheckBox cbCMVNeg;
		private System.Windows.Forms.Button btnClearAntigens;
		private System.Windows.Forms.ListView lstAntigens;
		private System.Windows.Forms.Label lblRBCAntigen;
		private System.Windows.Forms.ColumnHeader columnHeader8;
		private System.Windows.Forms.ColumnHeader columnHeader6;
		private System.Windows.Forms.TextBox txtAntigen;
		private System.Windows.Forms.Timer timer1;
		private System.ComponentModel.IContainer components;
		private System.Windows.Forms.ToolTip toolTip1;

		#endregion

		#region Control Events

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Button Click event. Clears all antigen information.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnClearAntigens_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();
			//
			if (this._allowUpdate)
			{
				this.antigensChanged = true;
				ClearAntigens(false);
			}
			else
			{
				ClearAntigens(true);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Textbox TextChanged event. Enables search timer if text entered
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void txtAntigen_TextChanged(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			//reset timer
			timer1.Enabled = false;
			//
			timer1.Enabled = (txtAntigen.Text.Length > 0);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Timer Tick event. Calls method to antigens matching entered text.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void timer1_Tick(object sender, System.EventArgs e)
		{
			timer1.Enabled = false;
			//
			DisplayMatchingAntigen();
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// ListView ItemCheck event. Checks or unchecks selected item, and ensures
		/// that an antigen is not checked as both positive AND negative. Also,
		/// adds change to antigen data table, for future filing.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void lstAntigens_ItemCheck(object sender, System.Windows.Forms.ItemCheckEventArgs e)
		{
			this.OnUserActivity();
			//
			ListViewItem lviCheck = lstAntigens.Items[e.Index];
			//
			// Check to see if updates are allowed
			if ((!(_allowUpdate)) && (!(this._isLoading)))
			{
				foreach(BOL.BloodUnitAntigen currBUA in this._preLoadedBloodUnitAntigens)
				{
					if(currBUA.AntigenTypeId == System.Convert.ToInt32(lviCheck.Tag.ToString()))
					{
						e.NewValue = e.CurrentValue;
						lviCheck.BackColor = Color.FromKnownColor(System.Drawing.KnownColor.Control);
						return;
					}
				}
			}
			//
			if (e.NewValue == CheckState.Checked)
			{
				// Antigen checked. If positive, deselect negative or vice versa 
				if ((lviCheck.SubItems[1].Text.IndexOf("NEG") > 0) && (lstAntigens.Items[e.Index+1].Checked))
				{
					lstAntigens.Items[e.Index+1].Checked = false;
					

				}
				if ((lviCheck.SubItems[1].Text.IndexOf("POS") > 0) && (lstAntigens.Items[e.Index-1].Checked))
				{
					lstAntigens.Items[e.Index-1].Checked = false;
					
				}
				//
				this._antigenChangedChecked = true;
			}
			//
			BOL.BloodUnitAntigen bloodUnitAntigen = new BOL.BloodUnitAntigen();
			bloodUnitAntigen.IsNew = true;
			bloodUnitAntigen.BloodUnitGuid = this.bloodUnitGUID;
			bloodUnitAntigen.AntigenTypeId = System.Convert.ToInt32(lviCheck.Tag.ToString());
			bloodUnitAntigen.Positive = (lviCheck.SubItems[1].Text.IndexOf("POS") > 0);
			//
			// 07/22/2004 D. Askew: Get row version from pre-loaded list, 
			// if available, in order to re-activate an antigen
			foreach (BOL.BloodUnitAntigen preLoadedAntigen in this._preLoadedBloodUnitAntigens)
			{
				if(preLoadedAntigen.AntigenTypeId == bloodUnitAntigen.AntigenTypeId)
				{
					bloodUnitAntigen.IsNew = false;
					bloodUnitAntigen.RowVersion = preLoadedAntigen.RowVersion;
					break;
				}
			}
			//
			// if unchecked and and companion unchecked, set mark for deletion
			if (e.NewValue == CheckState.Unchecked)
			{
				if (((lviCheck.SubItems[1].Text.IndexOf("NEG") > 0) && (!lstAntigens.Items[e.Index+1].Checked)) ||
					((lviCheck.SubItems[1].Text.IndexOf("POS") > 0) && (!lstAntigens.Items[e.Index-1].Checked)))
				{
					bloodUnitAntigen.RecordStatusCode = Common.RecordStatusCode.Inactive;
				}
				//
				this._antigenChangedChecked = false;
				if (Common.LogonUser.IsVisuallyImpairedUser && this._isLoading == false)
				{
					this.SpeakMessage( "unchecked" );	
				}
			}
			else
			{
				bloodUnitAntigen.RecordStatusCode = Common.RecordStatusCode.Active;
				if (Common.LogonUser.IsVisuallyImpairedUser && this._isLoading == false)
				{
					this.SpeakMessage( "checked" );	
				}
			}
			// Save the antigen that was updated and call OnAntigenUpdated()
			this._antigenChangedTypeId = bloodUnitAntigen.AntigenTypeId;
			this._antigenChangedType = bloodUnitAntigen.Positive ? _antigenTypePos : _antigenTypeNeg;
			this.OnAntigenUpdated();
			//
			// This code prevents the creation of multiple rows if an item is continually checked/ unchecked
			foreach(Object obj in this._bloodUnitAntigens)
			{
				BOL.BloodUnitAntigen currBloodUnitAntigen = (BOL.BloodUnitAntigen) obj;
				if(currBloodUnitAntigen.BloodUnitGuid == this.bloodUnitGUID &&
					currBloodUnitAntigen.AntigenTypeId == System.Convert.ToInt32(lviCheck.Tag.ToString()) &&
					currBloodUnitAntigen.Positive == bloodUnitAntigen.Positive)
				{
					this._bloodUnitAntigens.Remove(currBloodUnitAntigen);
					return;
				}
			}
			//
			if(!this._bloodUnitAntigens.Contains(bloodUnitAntigen))
			{
				this._bloodUnitAntigens.Add(bloodUnitAntigen);
			}
			//
			this.antigensChanged = true;
			this.IsDirty = true;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Checkbox CheckedChanged event. Sets CMV test result according to check value. 
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cbCMVNeg_CheckedChanged(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (cbCMVNeg.Checked)
			{
				this.cmvNeg.TestResultId = "N";
			}
			else
			{
				this.cmvNeg.TestResultId = "P";
			}
			this.cmvNeg.IsDirty = true;
			this.IsDirty = true;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Checkbox CheckedChanged event. Sets Sickle Cell test result according to check value.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cbSickleCellScreen_CheckedChanged(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (cbSickleCellScreen.Checked)
			{
				this.sickleCellNeg.TestResultId = "N";
			}
			else
			{
				this.sickleCellNeg.TestResultId = "P";
			}
			this.sickleCellNeg.IsDirty = true;
			this.IsDirty = true;
		}

		#endregion

		#region Events

		/// <summary>
		/// Raises is dirty changed event
		/// </summary>
		protected virtual void OnIsDirtyChanged()
		{
			if(IsDirtyChanged != null)
			{
				IsDirtyChanged(this, EventArgs.Empty);
			}
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>04/21/2006</CreationDate>
		/// <summary>
		/// Raises the antigen updated event
		/// </summary>
		protected virtual void OnAntigenUpdated()
		{
			if(AntigenUpdated != null)
			{
				AntigenUpdated(this, EventArgs.Empty);
			}
		}

		#endregion

		#region Constructors

		//private const int SICKLE_CELL_TEST_ID = 52;
		//private const int CMV_TEST_ID = 212;

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		/// <summary>
		/// SpecialTesting user control class constructor. Populates list view with all POS and NEG Antigen values.
		/// </summary>
		public SpecialTesting()
		{
			InitializeComponent();
			//
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#endregion

		#region Component Designer generated code

		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(SpecialTesting));
			this.grpSpecial = new System.Windows.Forms.GroupBox();
			this.txtAntigen = new System.Windows.Forms.TextBox();
			this.cbSickleCellScreen = new System.Windows.Forms.CheckBox();
			this.cbCMVNeg = new System.Windows.Forms.CheckBox();
			this.btnClearAntigens = new System.Windows.Forms.Button();
			this.lstAntigens = new System.Windows.Forms.ListView();
			this.columnHeader8 = new System.Windows.Forms.ColumnHeader();
			this.columnHeader6 = new System.Windows.Forms.ColumnHeader();
			this.lblRBCAntigen = new System.Windows.Forms.Label();
			this.timer1 = new System.Windows.Forms.Timer(this.components);
			this.toolTip1 = new System.Windows.Forms.ToolTip(this.components);
			this.grpSpecial.SuspendLayout();
			this.SuspendLayout();
			// 
			// grpSpecial
			// 
			this.grpSpecial.AccessibleDescription = "Special Testing";
			this.grpSpecial.AccessibleName = "Special Testing";
			this.grpSpecial.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpSpecial.Controls.Add(this.txtAntigen);
			this.grpSpecial.Controls.Add(this.cbSickleCellScreen);
			this.grpSpecial.Controls.Add(this.cbCMVNeg);
			this.grpSpecial.Controls.Add(this.btnClearAntigens);
			this.grpSpecial.Controls.Add(this.lstAntigens);
			this.grpSpecial.Controls.Add(this.lblRBCAntigen);
			this.grpSpecial.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.grpSpecial.Location = new System.Drawing.Point(0, 0);
			this.grpSpecial.Name = "grpSpecial";
			this.grpSpecial.Size = new System.Drawing.Size(176, 164);
			this.grpSpecial.TabIndex = 0;
			this.grpSpecial.TabStop = false;
			this.grpSpecial.Text = "Special Testing";
			// 
			// txtAntigen
			// 
			this.txtAntigen.AccessibleDescription = "R B C Antigens";
			this.txtAntigen.AccessibleName = "R B C Antigens";
			this.txtAntigen.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtAntigen.Location = new System.Drawing.Point(8, 80);
			this.txtAntigen.Name = "txtAntigen";
			this.txtAntigen.Size = new System.Drawing.Size(164, 20);
			this.txtAntigen.TabIndex = 3;
			this.txtAntigen.Text = "";
			this.toolTip1.SetToolTip(this.txtAntigen, "Enter a partial antigen name to find it in the list below");
			this.txtAntigen.TextChanged += new System.EventHandler(this.txtAntigen_TextChanged);
			// 
			// cbSickleCellScreen
			// 
			this.cbSickleCellScreen.AccessibleDescription = "Sickle Cell Negative";
			this.cbSickleCellScreen.AccessibleName = "Sickle Cell Negative";
			this.cbSickleCellScreen.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.cbSickleCellScreen.CheckAlign = System.Drawing.ContentAlignment.MiddleRight;
			this.cbSickleCellScreen.Location = new System.Drawing.Point(8, 38);
			this.cbSickleCellScreen.Name = "cbSickleCellScreen";
			this.cbSickleCellScreen.Size = new System.Drawing.Size(164, 24);
			this.cbSickleCellScreen.TabIndex = 2;
			this.cbSickleCellScreen.Text = "Sic&kle Cell Negative";
			this.toolTip1.SetToolTip(this.cbSickleCellScreen, "Check box to indicate that blood unit has been tested as Sickle Cell Negative");
			this.cbSickleCellScreen.CheckedChanged += new System.EventHandler(this.cbSickleCellScreen_CheckedChanged);
			// 
			// cbCMVNeg
			// 
			this.cbCMVNeg.AccessibleDescription = "C M V Negative";
			this.cbCMVNeg.AccessibleName = "C M V Negative";
			this.cbCMVNeg.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.cbCMVNeg.CheckAlign = System.Drawing.ContentAlignment.MiddleRight;
			this.cbCMVNeg.Location = new System.Drawing.Point(8, 16);
			this.cbCMVNeg.Name = "cbCMVNeg";
			this.cbCMVNeg.Size = new System.Drawing.Size(164, 24);
			this.cbCMVNeg.TabIndex = 1;
			this.cbCMVNeg.Text = "CM&V Negative";
			this.toolTip1.SetToolTip(this.cbCMVNeg, "Check box to indicate that blood unit has been tested as CMV Negative");
			this.cbCMVNeg.CheckedChanged += new System.EventHandler(this.cbCMVNeg_CheckedChanged);
			// 
			// btnClearAntigens
			// 
			this.btnClearAntigens.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.btnClearAntigens.FlatStyle = System.Windows.Forms.FlatStyle.Popup;
			this.btnClearAntigens.Image = ((System.Drawing.Image)(resources.GetObject("btnClearAntigens.Image")));
			this.btnClearAntigens.Location = new System.Drawing.Point(156, 64);
			this.btnClearAntigens.Name = "btnClearAntigens";
			this.btnClearAntigens.Size = new System.Drawing.Size(16, 16);
			this.btnClearAntigens.TabIndex = 4;
			this.toolTip1.SetToolTip(this.btnClearAntigens, "Click to deselect all antigens in the list");
			this.btnClearAntigens.Click += new System.EventHandler(this.btnClearAntigens_Click);
			// 
			// lstAntigens
			// 
			this.lstAntigens.AccessibleDescription = "R B C Antigens";
			this.lstAntigens.AccessibleName = "R B C Antigens";
			this.lstAntigens.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lstAntigens.CheckBoxes = true;
			this.lstAntigens.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
																						  this.columnHeader8,
																						  this.columnHeader6});
			this.lstAntigens.FullRowSelect = true;
			this.lstAntigens.HeaderStyle = System.Windows.Forms.ColumnHeaderStyle.None;
			this.lstAntigens.HideSelection = false;
			this.lstAntigens.Location = new System.Drawing.Point(8, 100);
			this.lstAntigens.MultiSelect = false;
			this.lstAntigens.Name = "lstAntigens";
			this.lstAntigens.Size = new System.Drawing.Size(164, 56);
			this.lstAntigens.TabIndex = 4;
			this.toolTip1.SetToolTip(this.lstAntigens, "Check box to associate antigen with blood unit");
			this.lstAntigens.View = System.Windows.Forms.View.Details;
			this.lstAntigens.SelectedIndexChanged += new System.EventHandler(this.lstAntigens_SelectedIndexChanged);
			this.lstAntigens.ItemCheck += new System.Windows.Forms.ItemCheckEventHandler(this.lstAntigens_ItemCheck);
			// 
			// columnHeader8
			// 
			this.columnHeader8.Width = 25;
			// 
			// columnHeader6
			// 
			this.columnHeader6.Width = 70;
			// 
			// lblRBCAntigen
			// 
			this.lblRBCAntigen.AccessibleDescription = "R B C Antigens";
			this.lblRBCAntigen.AccessibleName = "R B C Antigens";
			this.lblRBCAntigen.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lblRBCAntigen.Location = new System.Drawing.Point(8, 64);
			this.lblRBCAntigen.Name = "lblRBCAntigen";
			this.lblRBCAntigen.Size = new System.Drawing.Size(164, 16);
			this.lblRBCAntigen.TabIndex = 35;
			this.lblRBCAntigen.Text = "RBC Antigens";
			this.lblRBCAntigen.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// timer1
			// 
			this.timer1.Tick += new System.EventHandler(this.timer1_Tick);
			// 
			// toolTip1
			// 
			this.toolTip1.ShowAlways = true;
			// 
			// SpecialTesting
			// 
			this.Controls.Add(this.grpSpecial);
			this.Name = "SpecialTesting";
			this.Size = new System.Drawing.Size(176, 164);
			this.grpSpecial.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		#region Properties

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>04/22/2006</CreationDate>
		/// <summary>
		/// Gets the updated antigen's type id
		/// </summary>
		public int AntigenChangedTypeId
		{
			get
			{
				return this._antigenChangedTypeId;
			}
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>04/21/2006</CreationDate>
		/// <summary>
		/// Gets the updated antigen's type (POS or NEG)
		/// </summary>
		public string AntigenChangedType
		{
			get
			{
				return this._antigenChangedType != null ? this._antigenChangedType : string.Empty;
			}
		}

		///<Developers>
		///	<Developer>D. Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>04/22/2006</CreationDate>
		/// <summary>
		/// Has the changed antigen been checked or un-checked?
		/// </summary>
		public bool AntigenChangedChecked
		{
			get
			{
				return this._antigenChangedChecked;
			}
		}
	
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Gets or Sets the blood unit unique identifier. 
		/// Setting may retrieve CMV, Sickle Cell and Antigen test results for the blood unit.
		/// </summary>
		public Guid BloodUnitGuid
		{
			get
			{
				return this.bloodUnitGUID;
			}
			set
			{
				this.bloodUnitGUID = value;
				if (IsRetrievable)
				{
					GetCmvStatus();
					GetSickleCellStatus();
					GetAntigens();
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Gets or sets the division code.
		/// Setting may retrieve CMV, Sickle Cell and Antigen test results for the blood unit.
		/// </summary>
		public string DivisionCode
		{
			get
			{
				return this.divisionCode;
			}
			set
			{
				if (value.Length > 5)
				{
					throw new ArgumentException("Invalid Division Code: "+value);
				}
				this.divisionCode = value;
				if (IsRetrievable)
				{
					GetCmvStatus();
					GetSickleCellStatus();
					GetAntigens();
				}
			}
		}

		/// <summary>
		/// Gets or sets the property indicating whether a user can un-check
		/// previously-checked items or not.
		/// </summary>
		public bool AllowUpdate
		{
			get
			{
				return this._allowUpdate;
			}
			set
			{
				this._allowUpdate = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Gets value indicating whether to sufficient information has been entered to retrieve test results 
		/// </summary>
		/// <returns></returns>
		private bool IsRetrievable
		{
			get
			{
				if ((this.bloodUnitGUID == Guid.Empty) || (this.divisionCode.Length == 0))
				{
					return false;
				}
				else
				{
					return true;
				}
			}
		}
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/5/2002</CreationDate>
		///
		/// <summary>
		/// Has the object been changed
		/// </summary>
		public bool IsDirty
		{
			get
			{ 
				return _isDirty; 
			}
			set 
			{ 
				_isDirty = value;
				this.OnIsDirtyChanged();
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Gets or Sets the text contained in the user control.
		/// </summary>
		[
		Category("Appearance"),
		Description("The text contained in the control"),
		Bindable(false),
		Browsable(true),
		]
		new public string Text
		{
			get
			{
				return this.grpSpecial.Text;
			}
			set
			{
				this.grpSpecial.Text = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Override for the control Docking style property
		/// </summary>
		[
		Category("Layout"),
		Description("Docking style"),
		Bindable(false),
		Browsable(true),
		]
		public override DockStyle Dock 
		{
			get { return base.Dock; }
			set { base.Dock = value; }
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Override for the control's anchor style property
		/// </summary>
		[
		Category("Layout"),
		Description("Anchor style"),
		Bindable(false),
		Browsable(true),
		]
		public override AnchorStyles Anchor
		{
			get { return base.Anchor; }
			set { base.Anchor = value; }
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Gets the value indicating whether a blood unit has tested as CMV Negative
		/// </summary>
		public bool CmvNegative
		{
			set
			{
				this.cbCMVNeg.Checked = value;
			}
			get 
			{
				return (this.cbCMVNeg.Checked); 
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		/// <summary>
		/// Gets CMV blood unit test object
		/// </summary>
		public BOL.BloodUnitTest CmvTest
		{
			get
			{
				return this.cmvNeg;
			}
		}

		/// <summary>
		/// BR 1.53 uses this it enables or disables the sickle cell
		/// checkbox.
		/// </summary>
		public bool SickleCellEnabled
		{
			get
			{
				return this.cbSickleCellScreen.Enabled;
			}
			set
			{
				this.cbSickleCellScreen.Enabled = value;
			}
		}

		/// <summary>
		/// BR 1.53 uses this it enables or disables the CMV
		/// checkbox.
		/// </summary>
		public bool CmvEnabled
		{
			get
			{
				return this.cbCMVNeg.Enabled;
			}
			set
			{
				this.cbCMVNeg.Enabled = value;
			}
		}

		/// <summary>
		/// BR 1.53 uses this it enables or disables the antigens list
		/// </summary>
		public bool AntigensEnabled
		{
			get
			{
				return this.lstAntigens.Enabled;
			}
			set
			{
				this.lstAntigens.Enabled = value;
				this.txtAntigen.Enabled = value;
				this.btnClearAntigens.Enabled = value;
				if(value == false)
				{
					foreach(ListViewItem lvi in lstAntigens.Items)
					{
						lvi.BackColor = Color.FromKnownColor(System.Drawing.KnownColor.Control);
					}
				}
				else
				{
					foreach(ListViewItem lvi in lstAntigens.Items)
					{
						lvi.BackColor = Color.FromKnownColor(System.Drawing.KnownColor.White);
					}
				}
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Gets the value indicating whether a blood unit has tested as Sickle Cell Negative
		/// </summary>
		public bool SickleCellNegative
		{
			set
			{
				this.cbSickleCellScreen.Checked = value;
			}
			get 
			{
				return (this.cbSickleCellScreen.Checked); 
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		/// <summary>
		/// Gets sickle cell blood unit test object
		/// </summary>
		public BOL.BloodUnitTest SickleCellTest
		{
			get
			{
				return this.sickleCellNeg;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		/// <summary>
		/// String array containing a list of the antigens that were selected using the tool
		/// </summary>
		public string[] SelectedAntigens
		{
			get
			{
				string[] _antigens = new string[this.lstAntigens.CheckedItems.Count];
				//
				int i = -1;
				foreach (ListViewItem lvi in lstAntigens.CheckedItems)
				{
					i++;
					_antigens[i] = lvi.SubItems[1].Text;
				}
				return _antigens;
			}
		}
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		/// <summary>
		/// Gets value indicating whether use has changed list of antigens.
		/// </summary>
		public bool AntigensChanged
		{
			get
			{
				return this.antigensChanged;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		/// <summary>
		/// Gets table of antigens
		/// </summary>
		public System.Collections.ArrayList AntigensList
		{
			get
			{
				return this._bloodUnitAntigens;
			}
		}

		#endregion

		#region Methods
		

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Public method to reset special testing results to initail values
		/// </summary>
		public void ResetSpecialTesting()
		{
			this.cmvNeg = new BOL.BloodUnitTest(this.bloodUnitGUID, (int)Common.TestType.CMV);
			this.sickleCellNeg = new BOL.BloodUnitTest(this.bloodUnitGUID, (int)Common.TestType.SickleCell);
			this.txtAntigen.Text = string.Empty;
			cbCMVNeg.Checked = false;
			cbSickleCellScreen.Checked = false;
			ClearAntigens(false);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method to clear all Antigen test results from display and from data table
		/// </summary>
		/// <param name="keepOld">Clear only newly entered values?</param>
		private void ClearAntigens(bool keepOld)
		{
			this._isLoading = true;
			lstAntigens.SelectedItems.Clear();
			foreach(ListViewItem lvi in lstAntigens.Items)
			{
				if (keepOld)
				{
					bool oldAntigen = false;
					foreach(BOL.BloodUnitAntigen currBUA in this._preLoadedBloodUnitAntigens)
					{
						if(currBUA.AntigenTypeId == System.Convert.ToInt32(lvi.Tag.ToString()))
						{
							oldAntigen = true;
						}
					}
					if (!(oldAntigen))
					{
						lvi.Checked = false;
						if(this.lstAntigens.Enabled)
						{
							lvi.BackColor = Color.FromKnownColor(System.Drawing.KnownColor.White);
						}
					}
				}
				else
				{
					lvi.Checked = false;
					if(this.lstAntigens.Enabled)
					{
						lvi.BackColor = Color.FromKnownColor(System.Drawing.KnownColor.White);
					}
				}
			}
			this._bloodUnitAntigens.Clear();
			//
			this._isLoading = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Public method to call business layer to load BloodUnitTest object with
		/// CMV Status for blood unit, and display results.
		/// </summary>
		public void GetCmvStatus()
		{
			DataTable _dt = this.GetTestResult( (int) Common.TestType.CMV );
			if (_dt.Rows.Count>0)
			{
				DataRow _dr = _dt.Rows[0];
				this.cmvNeg = new BOL.BloodUnitTest(
					new Guid(_dr[TABLE.BloodUnitTest.BloodUnitTestGuid].ToString()),
					new Guid(_dr[TABLE.BloodUnitTest.BloodUnitTestGuid].ToString()),
					_dr.IsNull(TABLE.BloodUnitTest.BloodTestTypeId) ? Int16.MinValue : System.Convert.ToInt16(_dr[TABLE.BloodUnitTest.BloodTestTypeId]),
					_dr.IsNull(TABLE.BloodUnitTest.TestDate) ? DateTime.MinValue : System.Convert.ToDateTime(_dr[TABLE.BloodUnitTest.TestDate].ToString()),
					_dr[TABLE.BloodUnitTest.TestResultId].ToString(),
					_dr[TABLE.BloodUnitTest.TestComments].ToString(),
					_dr[TABLE.BloodUnitTest.EntryMethodCode].ToString(),
					_dr[TABLE.BloodUnitTest.DivisionCode].ToString()
					);

				this.cmvNeg.RowVersion = (byte[])_dr[TABLE.BloodUnitTest.RowVersion];
				if (this.cmvNeg.TestResultId.Trim() == "N")
				{
					cbCMVNeg.Checked = true;
				}
				else
				{
					cbCMVNeg.Checked = false;
				}
				this.cmvNeg.IsNew = false;
				this.cmvNeg.IsDirty = false;
			}
			else
			{
				this.cmvNeg = new BOL.BloodUnitTest(Guid.NewGuid(),
					this.bloodUnitGUID,
					(int) Common.TestType.CMV,
					BOL.VBECSDateTime.GetDivisionCurrentDateTime(),
					"P",
					string.Empty,
					"M",
					this.divisionCode);
				cbCMVNeg.Checked = false;
				this.cmvNeg.IsNew = true;
				this.cmvNeg.IsDirty = false;

			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Public method to call business layer to load BloodUnitTest object with
		/// Sickle Cell Status for blood unit, and display results.
		/// </summary>
		public void GetSickleCellStatus()
		{
			DataTable _dt = this.GetTestResult( (int) Common.TestType.SickleCell );
			if (_dt.Rows.Count>0)
			{
				DataRow _dr = _dt.Rows[0];
				this.sickleCellNeg = new BOL.BloodUnitTest(
					new Guid(_dr[TABLE.BloodUnitTest.BloodUnitTestGuid].ToString()),
					new Guid(_dr[TABLE.BloodUnitTest.BloodUnitTestGuid].ToString()),
					_dr.IsNull(TABLE.BloodUnitTest.BloodTestTypeId) ? Int16.MinValue : System.Convert.ToInt16(_dr[TABLE.BloodUnitTest.BloodTestTypeId]),
					_dr.IsNull(TABLE.BloodUnitTest.TestDate) ? DateTime.MinValue : System.Convert.ToDateTime(_dr[TABLE.BloodUnitTest.TestDate].ToString()),
					_dr[TABLE.BloodUnitTest.TestResultId].ToString(),
					_dr[TABLE.BloodUnitTest.TestComments].ToString(),
					_dr[TABLE.BloodUnitTest.EntryMethodCode].ToString(),
					_dr[TABLE.BloodUnitTest.DivisionCode].ToString()
					);

				this.sickleCellNeg.RowVersion = (byte[])_dr[TABLE.BloodUnitTest.RowVersion];
				if (this.sickleCellNeg.TestResultId.StartsWith("N"))
				{
					cbSickleCellScreen.Checked = true;
				}
				else
				{
					cbSickleCellScreen.Checked = false;
				}
				this.sickleCellNeg.IsNew = false;
				this.sickleCellNeg.IsDirty = false;
			}
			else
			{
				this.sickleCellNeg = new BOL.BloodUnitTest(Guid.NewGuid(),
					this.bloodUnitGUID,
					(int) Common.TestType.SickleCell ,
					BOL.VBECSDateTime.GetDivisionCurrentDateTime(),
					"P",
					string.Empty,
					"M",
					this.divisionCode);
				cbSickleCellScreen.Checked = false;
				this.sickleCellNeg.IsNew = true;
				this.sickleCellNeg.IsDirty = false;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private function which returns data table containing test results for a blood unit
		/// for a given blood test type.
		/// </summary>
		/// <param name="bloodTestTypeID"></param>
		/// <returns>Data table containing blood test result</returns>
		private DataTable GetTestResult(int bloodTestTypeID)
		{
			DataTable _dt;
			try
			{
				_dt = BOL.BloodUnitTest.GetBloodUnitTestResult(this.bloodUnitGUID,bloodTestTypeID,this.divisionCode);
				return _dt;
			}
			catch (Exception ex)
			{
				MessageBox.Show(ex.Message,"VBECS - Error",MessageBoxButtons.OK,MessageBoxIcon.Exclamation);
				return null;
			}
			
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method calls business layer to get antigen results for a blood unit,
		/// and display results in list view
		/// </summary>
		private void GetAntigens()
		{
			try
			{
				// 07/22/2004 D. Askew: Modified to get all antigens 
				// (both active and inactive ones) for a blood unit in order 
				// to keep track of the row version for antigens that need 
				// to be updated.
				SelectAntigens(BOL.BloodUnitAntigen.GetAllAntigensForBloodUnitList(this.bloodUnitGUID,this.divisionCode));
				this._bloodUnitAntigens.Clear();
				this.antigensChanged = false;
			}
			catch (Exception e)
			{
				MessageBox.Show(e.Message,"VBECS - Error",MessageBoxButtons.OK,MessageBoxIcon.Exclamation);
			}
		}
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/10/2002</CreationDate>
		/// <summary>
		/// GetSpecialTestsString
		/// </summary>
		/// <returns></returns>
		public string GetSpecialTestsString()
		{
			string specialTests = string.Empty;

			if (this.IsDirty)
			{
				System.Text.StringBuilder sb = new System.Text.StringBuilder();
				
				//CMV					
				if (this.CmvNegative)
				{
					sb.Append("CMV Negative/");
				}
		
				//Sickle Cell
				if (this.SickleCellNegative)
				{
					sb.Append("Sickle Cell Negative/");
				}

				//Get antigens
				if (this.SelectedAntigens.Length > 0)
				{
					for (int i = 1; i <= this.SelectedAntigens.Length; i++)
					{
						sb.Append(this.SelectedAntigens[i-1]);
						sb.Append("/");
					}
				}

				//Remove last "/"
				if (sb.Length > 0)
				{
					sb.Remove(sb.Length-1,1);
				}

				specialTests = sb.ToString();
			}

			return specialTests;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Public method to check a given data table of antigens in the list view display
		/// </summary>
		/// <param name="bloodUnitAntigens"></param>
		public void SelectAntigens(System.Collections.ArrayList bloodUnitAntigens)
		{
			this._isLoading = true;
			this._preLoadedBloodUnitAntigens = bloodUnitAntigens;
			
			foreach (Object obj in bloodUnitAntigens)
			{
				BOL.BloodUnitAntigen bloodUnitAntigen = (BOL.BloodUnitAntigen) obj;

				// 07/22/2004 D. Askew: Because we are now retrieving both inactive
				// and inactive antigens, make sure only active ones are selected
				if(bloodUnitAntigen.RecordStatusCode == Common.RecordStatusCode.Active)
				{
					foreach(ListViewItem _lvi in lstAntigens.Items)
					{
						if (bloodUnitAntigen.AntigenTypeId.ToString() == _lvi.Tag.ToString())
						{
							if (bloodUnitAntigen.Positive && (_lvi.SubItems[1].Text.IndexOf("POS") > -1))
							{
								//CR 2302
								this._bloodUnitAntigens.Add(bloodUnitAntigen);
								_lvi.Checked = true;
								if (!(this._allowUpdate))
								{
									_lvi.BackColor = Color.FromKnownColor(System.Drawing.KnownColor.Control);
									lstAntigens.Items[_lvi.Index-1].BackColor = Color.FromKnownColor(System.Drawing.KnownColor.Control);
								}
								break;
							}
							if (!bloodUnitAntigen.Positive && (_lvi.SubItems[1].Text.IndexOf("NEG") > -1))
							{
								//CR 2302
								this._bloodUnitAntigens.Add(bloodUnitAntigen);
								_lvi.Checked = true;
								if (!(this._allowUpdate))
								{
									_lvi.BackColor = Color.FromKnownColor(System.Drawing.KnownColor.Control);
									lstAntigens.Items[_lvi.Index+1].BackColor = Color.FromKnownColor(System.Drawing.KnownColor.Control);
								}
								break;
							}
						}
					}
				}
			}
			_isLoading = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method to search entries in listview and ensure that values matching
		/// text entered are visible to the user.
		/// </summary>
		private void DisplayMatchingAntigen()
		{
			foreach(ListViewItem lviAntigen in lstAntigens.Items)
			{
				//find exact match first
				if (lviAntigen.SubItems[1].Text == txtAntigen.Text)
				{
					lviAntigen.Selected = true;
					lviAntigen.EnsureVisible();
					break;
				}
				//find partial match
				if (lviAntigen.SubItems[1].Text.StartsWith(txtAntigen.Text))
				{
					lviAntigen.Selected = true;
					lviAntigen.EnsureVisible();
					break;
				}
			}
		}

		private static bool Include(string[] columnList,string comparisonId)
		{
			foreach(string col in columnList) 
			{
				if(col == comparisonId) 
				{
					return false;
				}
			}
			return true;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/28/2002</CreationDate>
		/// <summary>
		/// Public method which pre-loads all antigens into the list for selection.
		/// Should always be called after control is initialized  TT_23.01
		/// </summary>
		public void LoadAntigenList()
		{
			try
			{
				this._bloodUnitAntigens = new System.Collections.ArrayList();
				this._preLoadedBloodUnitAntigens = new System.Collections.ArrayList();

				lstAntigens.Items.Clear();

				BOL.AntibodyType at = new BOL.AntibodyType();
				
				DataTable _antigenTypes = BOL.AntibodyType.GetAntibodyType();

				foreach (DataRow _dr in _antigenTypes.Rows)
				{
					if ((_dr[TABLE.AntigenType.AntigenTypeName].ToString().Length > 0) && (_dr[TABLE.AntigenType.AntigenTypeName].ToString().ToUpper() != "NONE"))
					{
						//TT_23.01
						const string excludeAntibodyTypeIds = "11,61,20,83,79,25,84,6,90,43,72,35,85,88,1,4,82,39,89,23,49,64";
						if(Include(excludeAntibodyTypeIds.Split(','), _dr[TABLE.AntibodyType.AntibodyTypeId].ToString()))
						{
							ListViewItem _lviNeg = new ListViewItem();
							_lviNeg.SubItems.Add(_dr[TABLE.AntigenType.AntigenTypeName].ToString() + " NEG");
							_lviNeg.Tag = _dr[TABLE.AntigenType.AntigenTypeId].ToString();
							lstAntigens.Items.Add(_lviNeg);
							if(!lstAntigens.Enabled)
							{
								_lviNeg.BackColor =Color.FromKnownColor(System.Drawing.KnownColor.Control);
							}
							else
							{
								_lviNeg.BackColor =Color.FromKnownColor(System.Drawing.KnownColor.White);
							}
							//
							ListViewItem _lviPos = new ListViewItem();
							_lviPos.SubItems.Add(_dr[TABLE.AntigenType.AntigenTypeName].ToString() + " POS");
							_lviPos.Tag = _dr[TABLE.AntigenType.AntigenTypeId].ToString();
							lstAntigens.Items.Add(_lviPos);
							if(!lstAntigens.Enabled)
							{
								_lviPos.BackColor =Color.FromKnownColor(System.Drawing.KnownColor.Control);
							}
							else
							{
								_lviPos.BackColor =Color.FromKnownColor(System.Drawing.KnownColor.White);
							}
						}
						//
					}
				}
			}
			catch (Exception ex)
			{
				MessageBox.Show("Error initializing Antigen information: \n\n"+ex.Message,"VBECS - Error",MessageBoxButtons.OK,MessageBoxIcon.Exclamation);
			}
		}
		
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/16/2003</CreationDate>
		/// <summary>
		/// Public method to reload antigen selections from a hashtable
		/// </summary>
		/// <param name="antigens">Antigens hashtable</param>
		public void ReloadAntigens(Hashtable antigens)
		{
			this.cbCMVNeg.Checked = (bool) antigens[ ((int)Common.TestType.CMV).ToString() ];
			this.cbSickleCellScreen.Checked = (bool) antigens[ ((int)Common.TestType.SickleCell).ToString() ];
			// The antigens Hashtables has a listing of all of the antigens and a boolean to indicate
			// whether they have been selected or not.
				
			lstAntigens.Items.Clear();
			DataTable _antigenTypes = BOL.AntibodyType.GetAntibodyType();

			foreach (DataRow _dr in _antigenTypes.Rows)
			{
			
				if ((_dr[TABLE.AntigenType.AntigenTypeName].ToString().Length > 0) && (_dr[TABLE.AntigenType.AntigenTypeName].ToString().ToUpper() != "NONE"))
				{
					//
					ListViewItem _lviNeg = new ListViewItem();
					_lviNeg.SubItems.Add(_dr[TABLE.AntigenType.AntigenTypeName].ToString() + " NEG");
					_lviNeg.Tag = _dr[TABLE.AntibodyType.AntibodyTypeId].ToString();
					if (antigens !=null) 
					{
						if (antigens.ContainsKey(_dr[TABLE.AntibodyType.AntibodyTypeId].ToString().Trim() + "N")) 
						{
							_lviNeg.Checked = (bool) antigens[_dr[TABLE.AntibodyType.AntibodyTypeId].ToString().Trim() + "N"];
						}
					}
					lstAntigens.Items.Add(_lviNeg);
					
					//
					ListViewItem _lviPos = new ListViewItem();
					_lviPos.SubItems.Add(_dr[TABLE.AntigenType.AntigenTypeName].ToString() + " POS");
					_lviPos.Tag = _dr[TABLE.AntibodyType.AntibodyTypeId].ToString();
					if (antigens !=null) 
					{
						if (antigens.ContainsKey(_dr[TABLE.AntibodyType.AntibodyTypeId].ToString().Trim() + "P"))
						{
							_lviPos.Checked = (bool) antigens[_dr[TABLE.AntibodyType.AntibodyTypeId].ToString().Trim() + "P" ];
						}
					}
					lstAntigens.Items.Add(_lviPos);
					//
				}
			}

		}

		
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Public method to save to file the CMV testing status of a blood unit
		/// </summary>
		public void SaveCmvNegStatus()
		{
			if (this.cmvNeg.IsDirty)
			{
				if (this.cmvNeg.IsNew)
				{
					this.cmvNeg.SaveBloodUnitTest(System.Guid.Empty, Common.UpdateFunction.SpecialTestingSaveCmvNegStatus);
				}
				else
				{
					this.cmvNeg.UpdateTestResult(Common.UpdateFunction.SpecialTestingSaveCmvNegStatus);
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Public method to save to file the Sickle Cell testing status of a blood unit
		/// </summary>
		public void SaveSickleCellStatus()
		{
			if (this.sickleCellNeg.IsDirty)
			{
				if (this.sickleCellNeg.IsNew)
				{
					this.sickleCellNeg.SaveBloodUnitTest(System.Guid.Empty, Common.UpdateFunction.SpecialTestingSaveSickleCellStatus);
				}
				else
				{
					this.sickleCellNeg.UpdateTestResult(Common.UpdateFunction.SpecialTestingSaveSickleCellStatus);
				}
			}
		}

		#endregion

		private void lstAntigens_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			//508 Compliance
			this.SpeakListViewWithCheckBox(lstAntigens);
		}
	}

}

